//
// (c) Mark McIlroy June 2022. Author of the design of the Bitstream Tree.
//
//

// Bitstream tree.
//
// The maximum depth of an item in the tree is the number of bits in the item's key.
//
// For sorted output the key must be a text string, int, or floating point number converted to text and zero padded.
//
// Each item inserted into the tree must have a unique key.
//
// For text keys count the number of characters excluding the null as 'num_key_bytes'.
//
// For integer keys the highest byte should be first in the byte sequence.
//
// For insert/lookup only, any binary key can be used.


// Integer values are unsigned, as the tree processes keys bit-by-bit. To correctly sort negative integers add a bias,
// e.g. positive values in the range 5,000,000 to 10,000,0000 and negative values in the range 0 to 5,000,000
//
// bias = 0x7FFFFFFFFFFFFFFF;
//
// Input key: input( x + bias );
//
// for/next loop
//
// x = bst_nptr.get_intkey();
//
// x -= bias;




#include "bst_tree_c_version.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>



//bst_tree_node *c_bst_tree_first_item2( bst_tree_node *bst_nptr, int *found );

void c_bst_tree_free2( bst_tree_node *nptr );
int bget_bit1( char const *key_bits, int pos );
int bget_bit( char const *key_bits, int pos, int pos1 );


#define MAX_DEBUG_TREE_ITEMS	100000

#ifdef DEBUG	
	int gnode_number;
	bst_tree_node *node_ptr[MAX_DEBUG_TREE_ITEMS];
#endif



bst_tree *c_bst_init2( int options )
{
	bst_tree *tptr;
	bst_tree_node *nptr;

	tptr = malloc( sizeof( bst_tree ) );
	
	nptr = malloc( sizeof( bst_tree_node ) );  

	tptr->options = options;
	tptr->total_items_in_tree = 0;
	tptr->unique_items_in_tree = 0;
	
#ifdef DEBUG	
	gnode_number = 0;
	nptr->node_number = 0;
	node_ptr[gnode_number++] = nptr;
	
	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif
	
	memset( nptr, 0, sizeof( bst_tree_node ) );
	
/*
	nptr->key_bits = NULL;
	nptr->num_key_bytes = 0;
	nptr->num_key_bits_this_node = 0;
	nptr->is_set = 0;
	nptr->duplicate_count = 0;
	nptr->data_ptr = NULL;
	nptr->lchild = NULL;
	nptr->rchild = NULL;
	nptr->parent_ptr = NULL;
*/
	
	tptr->root_node = nptr;
	
	return (tptr);
}


void bst_node_init( bst_tree_node *bst_nptr )
{
	memset( bst_nptr, 0, sizeof( bst_tree_node ) );

/*
	bst_nptr->key_bits = NULL;

	bst_nptr->num_key_bytes = 0;
	bst_nptr->num_key_bits_this_node = 0;

	bst_nptr->duplicate_count = 0;

	bst_nptr->data_ptr = NULL;

	bst_nptr->is_set = false;
	
	bst_nptr->lchild = NULL;
	bst_nptr->rchild = NULL;
*/	
}



void bst_node_copy( bst_tree_node *bst_nptr2, bst_tree_node *bst_nptr )
{
	if (bst_nptr2->key_bits != NULL)
		free( bst_nptr2->key_bits );

	memcpy( bst_nptr2, bst_nptr, sizeof( bst_tree_node ) );

/*
	bst_nptr2->num_key_bytes = bst_nptr->num_key_bytes;
	bst_nptr2->num_key_bits_this_node = bst_nptr->num_key_bits_this_node;
	bst_nptr2->is_set = bst_nptr->is_set;
	bst_nptr2->duplicate_count = bst_nptr->duplicate_count;
	bst_nptr2->data_ptr = bst_nptr->data_ptr;
	bst_nptr2->lchild = bst_nptr->lchild;
	bst_nptr2->rchild = bst_nptr->rchild;
*/	
		
	bst_nptr2->key_bits = malloc( bst_nptr->num_key_bytes );
	
	memcpy( bst_nptr2->key_bits, bst_nptr->key_bits, bst_nptr->num_key_bytes );
}



//-------------------------------------------------------------------------------------
// Insert a data item into a Bitstream tree. Returns 0 on success, or 1 if a duplicate.
//-------------------------------------------------------------------------------------

int c_bst_insert( bst_tree *tree_ptr, char const *key_bits, int num_key_bytes, void *data_ptr )
{
	int bit, bit1;
	int num_input_key_bits;
	int tree_num_bits;
	int bit_pos;
	int bit_pos1;
	int bit_pos2;
	bst_tree_node *bst_nptr, *bst_nptr2;
	int finished;
	int result;

	if (tree_ptr == NULL)
	{
		printf( "Error: NULL tree pointer in c_bst_insert()\n" );
		exit(1);
	}

	if (key_bits == NULL)
	{
		printf( "Error: NULL key pointer in c_bst_insert()\n" );
		exit(1);
	}
	
	if (num_key_bytes == 0)
	{
		printf( "Error: Can't insert an item into a bst_tree with a key length of 0.\n" );
		exit(1);
	}
	
	bst_nptr = tree_ptr->root_node;
	
	
	num_input_key_bits = num_key_bytes * 8;

	bit_pos = 0;

	finished = false;

		// continue descent until reaching the end of the node's bit sequence, the insert key number of bits, or key mismatch

		// cases: insert key is shorter and matches
		//		  insert key is longer and matches
		//		  exact match of keys (i.e. a duplicate key or previously deleted item reinsert)
		//		  bit mismatch
		
	bit_pos1 = 0;
	bit_pos2 = 0x80;
	
	bit = bget_bit( key_bits, 0, 0x80 );
				
	if (bit == 0 && bst_nptr->lchild == NULL)
	{
		bst_nptr->lchild = malloc( sizeof( bst_tree_node ) );

#ifdef DEBUG	
	bst_nptr->lchild->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr->lchild;
	gnode_number++;
	
	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif

		bst_node_init( bst_nptr->lchild );
		
		bst_nptr->lchild->num_key_bits_this_node = num_input_key_bits;
		
		bst_nptr = bst_nptr->lchild;
		
		bst_nptr->parent_ptr = tree_ptr->root_node;
	}
	else	
 	if (bit == 1 && bst_nptr->rchild == NULL)	
	{
		bst_nptr->rchild = malloc( sizeof( bst_tree_node ) );

#ifdef DEBUG	
	bst_nptr->rchild->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr->rchild;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif

		bst_node_init( bst_nptr->rchild );
		
		bst_nptr->rchild->num_key_bits_this_node = num_input_key_bits;
		
		bst_nptr = bst_nptr->rchild;

		bst_nptr->parent_ptr = tree_ptr->root_node;
	}							
	else 
	{
		if (bit == 0)
			bst_nptr = bst_nptr->lchild;
		else
			bst_nptr = bst_nptr->rchild;
		
		tree_num_bits = bst_nptr->num_key_bits_this_node;
		
		while (! finished)
		{
			if (bit_pos == num_input_key_bits)					// past the end of the input key
			{													
																
			 	if (num_key_bytes == bst_nptr->num_key_bytes)


	// Case 1
				
	// past the end of the input key, input key the same size as a tree node, 
	// therefore an exact match (duplicate of an existing key).



					finished = true;
			}
			
			if (! finished)
			{
				if (bit_pos >= tree_num_bits)
				{
	
	// Case 2
				
	// match, and insert key is longer than key bits processed so far.
	// Add the key being inserted as a new leaf node (next bit 0).
					
					bit = bget_bit( key_bits, bit_pos1, bit_pos2 );
					
					if (bit == 0)
					{
						if (bst_nptr->lchild == NULL)
						{
							bst_nptr->lchild = malloc( sizeof( bst_tree_node ) );
							
#ifdef DEBUG	
	bst_nptr->lchild->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr->lchild;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif
							
							bst_node_init( bst_nptr->lchild );
							
							bst_nptr->lchild->num_key_bits_this_node = num_input_key_bits - bit_pos;

							bst_nptr->lchild->parent_ptr = bst_nptr;

							bst_nptr = bst_nptr->lchild;
							
							finished = true;
						}
						else
						{
							bst_nptr = bst_nptr->lchild;
							
							tree_num_bits += bst_nptr->num_key_bits_this_node; 
						}
					}
					else
					{
					
	// Case 3
				
	// match, and insert key is longer than key bits processed so far.
	// Add the key being inserted as a new leaf node (next bit 1).
					
						if (bst_nptr->rchild == NULL)
						{
							bst_nptr->rchild = malloc( sizeof( bst_tree_node ) );
					
#ifdef DEBUG	
	bst_nptr->rchild->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr->rchild;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif
					
							bst_node_init( bst_nptr->rchild );
							
							bst_nptr->rchild->num_key_bits_this_node = num_input_key_bits - bit_pos;

							bst_nptr->rchild->parent_ptr = bst_nptr;

							bst_nptr = bst_nptr->rchild;
							
							finished = true;
						}
						else
						{
							bst_nptr = bst_nptr->rchild;
							
							tree_num_bits += bst_nptr->num_key_bits_this_node; 
						}
					}
				}	
				else
				if (bit_pos >= num_input_key_bits)	
				{

	// Case 4
					
	// match, and insert key is shorter than currently stored key.
	// Split the current node and add the existing subtreee as a subtree (next bit 0).
								
					bit1 = bget_bit( bst_nptr->key_bits, bit_pos1, bit_pos2 );
	
					if (bit1 == 0)
					{
						bst_nptr2 = malloc( sizeof( bst_tree_node ) );

#ifdef DEBUG	
	bst_nptr2->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr2;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif

						bst_nptr2->key_bits = NULL;		// don't attempt to free uninitialised pointer

						bst_node_copy( bst_nptr2, bst_nptr );

						bst_nptr2->num_key_bits_this_node = tree_num_bits - bit_pos; 

						bst_nptr2->parent_ptr = bst_nptr;

						if (bst_nptr2->lchild != NULL)
							bst_nptr2->lchild->parent_ptr = bst_nptr2;
							 
						if (bst_nptr2->rchild != NULL)
							bst_nptr2->rchild->parent_ptr = bst_nptr2;

						bst_nptr->lchild = bst_nptr2;
						bst_nptr->rchild = NULL;

						bst_nptr->is_set = false;
						bst_nptr->num_key_bits_this_node -= bst_nptr2->num_key_bits_this_node;
	
						finished = true;
					}
					else
					{
					
	// Case 5
					
	// match, and the insert key is shorter than currently stored key.
	// Split the current node and add the existing subtreee as a subtree (next bit 1).
					
						bst_nptr2 = malloc( sizeof( bst_tree_node ) );

#ifdef DEBUG	
	bst_nptr2->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr2;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif

						bst_nptr2->key_bits = NULL;		// don't attempt to free uninitialised pointer

						bst_node_copy( bst_nptr2, bst_nptr );

						bst_nptr2->num_key_bits_this_node = tree_num_bits - bit_pos; 

						bst_nptr2->parent_ptr = bst_nptr;

						if (bst_nptr2->lchild != NULL)
							bst_nptr2->lchild->parent_ptr = bst_nptr2;
							 
						if (bst_nptr2->rchild != NULL)
							bst_nptr2->rchild->parent_ptr = bst_nptr2;

						bst_nptr->rchild = bst_nptr2;
						bst_nptr->lchild = NULL;

						bst_nptr->is_set = false;
						bst_nptr->num_key_bits_this_node -= bst_nptr2->num_key_bits_this_node;
	
						finished = true;
					}
				}
				else
				{
					bit = bget_bit( key_bits, bit_pos1, bit_pos2 );
	
					bit1 = bget_bit( bst_nptr->key_bits, bit_pos1, bit_pos2 );
	
					if (bit == bit1)	// progressing though the key, match
					{
						bit_pos++;
					
						bit_pos2 = bit_pos2 >> 1;
						
						if (bit_pos2 == 0)
						{
							bit_pos1++;
					
							bit_pos2 = 0x80;
						}
					}	
					else
					{						
					
	// Case 6

	// bit pattern mismatch.
	// Split the current node and add the key being inserted as a new leaf node (next bit 0).

						if (bit == 0)
						{
							bst_nptr2 = malloc( sizeof( bst_tree_node ) );
	
#ifdef DEBUG	
	bst_nptr2->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr2;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif
	
							bst_nptr2->key_bits = NULL;		// don't attempt to free uninitialised pointer
	
							bst_node_copy( bst_nptr2, bst_nptr );
	
							bst_nptr2->num_key_bits_this_node = tree_num_bits - bit_pos; 

							bst_nptr2->parent_ptr = bst_nptr;

							if (bst_nptr2->lchild != NULL)
								bst_nptr2->lchild->parent_ptr = bst_nptr2;
								 
							if (bst_nptr2->rchild != NULL)
								bst_nptr2->rchild->parent_ptr = bst_nptr2;

							bst_nptr->rchild = bst_nptr2;

							bst_nptr->lchild = malloc( sizeof( bst_tree_node ) );

#ifdef DEBUG	
	bst_nptr->lchild->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr->lchild;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif

							bst_node_init( bst_nptr->lchild );

							bst_nptr->lchild->num_key_bits_this_node = num_input_key_bits - bit_pos; 

							bst_nptr->lchild->parent_ptr = bst_nptr;

							bst_nptr->is_set = false;

							bst_nptr->num_key_bits_this_node -= bst_nptr2->num_key_bits_this_node;
							
							bst_nptr = bst_nptr->lchild;
		
							finished = true;
						}
						else
						
	// Case 7

	// bit pattern mismatch.
	// Split the current node and add the key being inserted as a new leaf node (next bit 1).

						
						{
							bst_nptr2 = malloc( sizeof( bst_tree_node ) );
	
#ifdef DEBUG	
	bst_nptr2->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr2;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif
	
							bst_nptr2->key_bits = NULL;		// don't attempt to free uninitialised pointer
	
							bst_node_copy( bst_nptr2, bst_nptr );
	
							bst_nptr2->num_key_bits_this_node = tree_num_bits - bit_pos; 

							bst_nptr2->parent_ptr = bst_nptr;

							if (bst_nptr2->lchild != NULL)
								bst_nptr2->lchild->parent_ptr = bst_nptr2;
								 
							if (bst_nptr2->rchild != NULL)
								bst_nptr2->rchild->parent_ptr = bst_nptr2;

							bst_nptr->lchild = bst_nptr2;

							bst_nptr->rchild = malloc( sizeof( bst_tree_node ) );

#ifdef DEBUG	
	bst_nptr->rchild->node_number = gnode_number;
	node_ptr[gnode_number] = bst_nptr->rchild;
	gnode_number++;

	if (gnode_number >= MAX_DEBUG_TREE_ITEMS) {printf( "BST too many debug items.\n" ); exit( 1 ); }
#endif

							bst_node_init( bst_nptr->rchild );

							bst_nptr->rchild->num_key_bits_this_node = num_input_key_bits - bit_pos; 

							bst_nptr->rchild->parent_ptr = bst_nptr;

							bst_nptr->is_set = false;

							bst_nptr->num_key_bits_this_node -= bst_nptr2->num_key_bits_this_node;
							
							bst_nptr = bst_nptr->rchild;
		
							finished = true;
						}
					}
				}
			}
		}
	}

	if (bst_nptr->is_set)
	{
		bst_nptr->duplicate_count++;

		tree_ptr->total_items_in_tree++;
		
		result = 1;
	}
	else
	{
		result = 0;
		
		tree_ptr->unique_items_in_tree++;

		tree_ptr->total_items_in_tree++;
		
		bst_nptr->duplicate_count = 0;
		
		bst_nptr->is_set = true;
		
		if (bst_nptr->key_bits != NULL)
			free( bst_nptr->key_bits );
			
		bst_nptr->key_bits = malloc( num_key_bytes );
		
		memcpy( bst_nptr->key_bits, key_bits, num_key_bytes );
		
		bst_nptr->num_key_bytes = num_key_bytes; 
	}
		
	bst_nptr->data_ptr = data_ptr;
		
	return (result);
}


//---------------------------------------------------------------------------------------------------
// Print a bst tree for debugging purposes. This code must be compiled with #define DEBUG 1 for this.
//---------------------------------------------------------------------------------------------------

#ifdef DEBUG	

void bst_tree_debug_print_tree()
{
	int i, j;
	char str[10000];

	printf( "node_number is_set key lchild rchild parent_ptr\n\n" );

	for (i=0; i < gnode_number; i++)
	{
		strncpy( str, node_ptr[i]->key_bits, node_ptr[i]->num_key_bytes );

		str[node_ptr[i]->num_key_bytes] = '\0';
	
		printf( "%4d ", node_ptr[i]->node_number );

		if (node_ptr[i]->is_set)
			printf( "is set      " );
		else
			printf( "is not set  " );

		printf( "%s ", str );
			
		if (node_ptr[i]->lchild == NULL)
			printf( "NULL " );
		else
		for (j=0; j < gnode_number; j++)
		{
			if (node_ptr[i]->lchild == node_ptr[j])
				printf( "%4d ", node_ptr[j]->node_number );
		}

		if (node_ptr[i]->rchild == NULL)
			printf( "NULL " );
		else
		for (j=0; j < gnode_number; j++)
		{
			if (node_ptr[i]->rchild == node_ptr[j])
				printf( "%4d ", node_ptr[j]->node_number );
		}

		if (node_ptr[i]->parent_ptr == NULL)
			printf( "NULL " );
		else
		for (j=0; j < gnode_number; j++)
		{
			if (node_ptr[i]->parent_ptr == node_ptr[j])
				printf( "%4d ", node_ptr[j]->node_number );
		}

		printf( "\n" );
	}
}

#endif




//-------------------------------------------------------------------------------------
// Search for an item in a tree. Returns a pointer to the tree node if found otherwise NULL.
//-------------------------------------------------------------------------------------

bst_tree_node *c_bst_search( bst_tree *tree_ptr, char const *key_bits, int num_key_bytes, int *found )
{
	int bit, bit1;
	int num_input_key_bits;
	int tree_num_bits;
	bst_tree_node *bst_nptr;
	int finished;
	bst_tree_node *result;

	if (tree_ptr == NULL)
	{
		printf( "Error: NULL tree pointer in c_bst_search()\n" );
		exit(1);
	}

	if (key_bits == NULL)
	{
		printf( "Error: NULL key pointer in c_bst_search()\n" );
		exit(1);
	}

	if (num_key_bytes == 0)
	{
		printf( "Error: Can't search for an item in a bst_tree with a key length of 0, c_bst_search()\n" );
		exit(1);
	}
	
	num_input_key_bits = num_key_bytes * 8;

	tree_num_bits = 0;

	*found = false;
	
	finished = false;
	
	bit = bget_bit1( key_bits, 0 );
				
	if (bit == 0)
		bst_nptr = tree_ptr->root_node->lchild; 
	else
		bst_nptr = tree_ptr->root_node->rchild; 

	if (bst_nptr == NULL)
		*found = false;
	else
	{
		while (! finished)
		{
			tree_num_bits += bst_nptr->num_key_bits_this_node; 
			
			if (num_input_key_bits < tree_num_bits) 
				finished = true;
			else			
			if (num_input_key_bits == tree_num_bits)
			{
				if (bst_nptr->is_set)
				{
				 	if (num_key_bytes == bst_nptr->num_key_bytes)
				 	{
				 	 	if (memcmp( key_bits, bst_nptr->key_bits, num_key_bytes ) == 0)
							*found = true;
					}
				}
				
				finished = true;
			}
			else
			{
				bit = bget_bit1( key_bits, tree_num_bits );
				
				if (bit == 0)
				{
					if (bst_nptr->lchild == NULL)
						finished = true;
					else
						bst_nptr = bst_nptr->lchild;
				}
				else
				{
					if (bst_nptr->rchild == NULL)
						finished = true;
					else
						bst_nptr = bst_nptr->rchild;
				}
			}
		}
	}
		 
	if (*found)
		result = bst_nptr->data_ptr;
	else
		result = NULL;
		
	return (result);
}


//-------------------------------------------------------------------------------------
// Returns true if the key was found in the tree.
//-------------------------------------------------------------------------------------

int c_bst_key_is_in_tree( bst_tree *tree_ptr, char const *key_bits, int num_key_bytes )
{
	int bit, bit1;
	int num_input_key_bits;
	int tree_num_bits;
	bst_tree_node *bst_nptr;
	int finished;
	bst_tree_node *result;
	int found;
	
	if (tree_ptr == NULL)
	{
		printf( "Error: NULL tree pointer in c_bst_key_is_in_tree()\n" );
		exit(1);
	}

	if (key_bits == NULL)
	{
		printf( "Error: NULL key pointer in c_bst_key_is_in_tree()\n" );
		exit(1);
	}

	if (num_key_bytes == 0)
	{
		printf( "Error: Can't search for an item in a bst_tree with a key length of 0, c_bst_key_is_in_tree()\n" );
		exit(1);
	}
	
	num_input_key_bits = num_key_bytes * 8;

	tree_num_bits = 0;

	found = false;
	
	finished = false;
	
	bit = bget_bit1( key_bits, 0 );
				
	if (bit == 0)
		bst_nptr = tree_ptr->root_node->lchild; 
	else
		bst_nptr = tree_ptr->root_node->rchild; 
	
	if (bst_nptr != NULL)
	{
		while (! finished)
		{
			tree_num_bits += bst_nptr->num_key_bits_this_node; 
			
			if (num_input_key_bits < tree_num_bits) 
				finished = true;
			else			
			if (num_input_key_bits == tree_num_bits)
			{
				if (bst_nptr->is_set)
				{
				 	if (num_key_bytes == bst_nptr->num_key_bytes)
				 	{
				 		if (memcmp( key_bits, bst_nptr->key_bits, num_key_bytes ) == 0)
							found = true;
					}
				}
				
				finished = true;
			}
			else
			{
				bit = bget_bit1( key_bits, tree_num_bits );
				
				if (bit == 0)
				{
					if (bst_nptr->lchild == NULL)
						finished = true;
					else
						bst_nptr = bst_nptr->lchild;
				}
				else
				{
					if (bst_nptr->rchild == NULL)
						finished = true;
					else
						bst_nptr = bst_nptr->rchild;
				}
			}
		}
	}
				 
	return (found);
}


//-------------------------------------------------------------------------------------
// Delete an item from a tree. Returns true if the item was found.
//-------------------------------------------------------------------------------------

int c_bst_delete_item( bst_tree *tree_ptr, char const *key_bits, int num_key_bytes )
{
	int bit, bit1;
	int num_input_key_bits;
	int tree_num_bits;
	int bit_pos;
	bst_tree_node *bst_nptr;
	int finished;
	bst_tree_node *result;
	int found;

	if (tree_ptr == NULL)
	{
		printf( "Error: NULL tree pointer in c_bst_delete_item()\n" );
		exit(1);
	}

	if (key_bits == NULL)
	{
		printf( "Error: NULL key pointer in c_bst_delete_item()\n" );
		exit(1);
	}

	if (num_key_bytes == 0)
	{
		printf( "Error: Can't delete an item from a bst_tree with a key length of 0, c_bst_delete_item()\n" );
		exit(1);
	}
	
	num_input_key_bits = num_key_bytes * 8;

	bit_pos = 0;

	found = false;
	
	finished = false;
	
	bit = bget_bit1( key_bits, 0 );
				
	if (bit == 0)
		bst_nptr = tree_ptr->root_node->lchild; 
	else
		bst_nptr = tree_ptr->root_node->rchild; 

	tree_num_bits = bst_nptr->num_key_bits_this_node;
			
	while (! finished)
	{
		if (bit_pos == num_input_key_bits)
		{
			if (bst_nptr->is_set)
			{
			 	if (num_key_bytes == bst_nptr->num_key_bytes && memcmp( key_bits, bst_nptr->key_bits, num_key_bytes ) == 0)
			 	{
			 		bst_nptr->is_set = false;
			 		
					found = true;
				}
			}
			
			finished = true;
		}
		
		if (! finished)
		{
			if (bit_pos >= tree_num_bits)
			{
				bit = bget_bit1( key_bits, bit_pos );
				
				if (bit == 0)
				{
					if (bst_nptr->lchild == NULL)
						finished = true;
					else
						bst_nptr = bst_nptr->lchild;
				}
				else
				{
					if (bst_nptr->rchild == NULL)
						finished = true;
					else
						bst_nptr = bst_nptr->rchild;
				}
				
				tree_num_bits += bst_nptr->num_key_bits_this_node; 
			}
		}
		
		bit_pos++;
	}
	
	if (found)
	{	
		tree_ptr->unique_items_in_tree--;

		tree_ptr->total_items_in_tree -= bst_nptr->duplicate_count + 1;
	}
	
	return (found);
}


	
//-------------------------------------------------------------------------------------
// Full scan of tree
//-------------------------------------------------------------------------------------
		
/*
void c_bst_prep_for_scan( bst_tree *tree_ptr, int ascending )
{
	bst_tree_node *curr_nptr;

	if (tree_ptr == NULL)
	{
		printf( "Error: null link passed to c_bst_prep_for_scan()" );
		exit(1);
	}

	curr_nptr = tree_ptr->root_node;
	
	if (ascending)
		c_bst_scan_asc( tree_ptr->root_node, &curr_nptr );
	else
		c_bst_scan_desc( tree_ptr->root_node, &curr_nptr );
	
	curr_nptr->next = NULL;
}

		
void c_bst_scan_asc( bst_tree_node *bst_nptr, bst_tree_node **curr_nptr )
{
	if (bst_nptr->is_set)
	{
		(*curr_nptr)->next = bst_nptr;
		*curr_nptr = bst_nptr;
	}

	if (bst_nptr->lchild != NULL)
		c_bst_scan_asc( bst_nptr->lchild, curr_nptr );
	
	if (bst_nptr->rchild != NULL)
		c_bst_scan_asc( bst_nptr->rchild, curr_nptr );
}


void c_bst_scan_desc( bst_tree_node *bst_nptr, bst_tree_node **curr_nptr )
{
	if (bst_nptr->rchild != NULL)
		c_bst_scan_desc( bst_nptr->rchild, curr_nptr );

	if (bst_nptr->lchild != NULL)
		c_bst_scan_desc( bst_nptr->lchild, curr_nptr );
	
	if (bst_nptr->is_set)
	{
		(*curr_nptr)->next = bst_nptr;
		*curr_nptr = bst_nptr;
	}
}
*/


void c_bst_free( bst_tree *tree_ptr )
{
	if (tree_ptr != NULL)
	{
		if (tree_ptr->root_node != NULL)
		{
			c_bst_tree_free2( tree_ptr->root_node );
		
			tree_ptr->root_node = NULL;
		}
		
		free( tree_ptr );
	}
}


void c_bst_tree_free2( bst_tree_node *nptr )
{
	if (nptr->lchild != NULL)
		c_bst_tree_free2( nptr->lchild );
	
	if (nptr->rchild != NULL)
		c_bst_tree_free2( nptr->rchild );
			
	if (nptr->key_bits != NULL)
		free( nptr->key_bits );
			
	free( nptr );
}


bst_tree_node *c_bst_tree_first_item( bst_tree *tree_ptr, int ascending, char *source_file_id_txt )
{
	bst_tree_node *bst_nptr;
	int finished;
	int found;

	if (tree_ptr == NULL)
	{
		printf( "NULL passed to c_bst_tree_first_item()" );
		exit( 1 );
	}

	if (ascending)
	{
		finished = false;

		bst_nptr = NULL;
		
//		if (tree_ptr->root_node->is_set)
//			bst_nptr = tree_ptr->root_node;
//		else

		if (tree_ptr->root_node->lchild != NULL)
			bst_nptr = tree_ptr->root_node->lchild;
		else
		if (tree_ptr->root_node->rchild != NULL)
			bst_nptr = tree_ptr->root_node->rchild;
				
		while (! finished)
		{
			if (bst_nptr == NULL)
				finished = true;
			else
			if (bst_nptr->is_set)
				finished = true;
			else
			{
				if (bst_nptr->lchild != NULL)
					bst_nptr = bst_nptr->lchild;
				else
				if (bst_nptr->rchild != NULL)
					bst_nptr = bst_nptr->rchild;
				else
				if (bst_nptr->is_set)
					finished = true;
				else
				{
					bst_nptr = c_bst_tree_next_item( bst_nptr, ascending );				
					finished = true;
				}
			}		
		}
	}
	else
	{
		bst_nptr = tree_ptr->root_node;
	
		while (! (bst_nptr->lchild == NULL && bst_nptr->rchild == NULL))
		{
			if (bst_nptr->rchild != NULL)
				bst_nptr = bst_nptr->rchild;
			else   
				bst_nptr = bst_nptr->lchild;   
		}
		
		if (bst_nptr != NULL)
		{ 
			if (! bst_nptr->is_set)
				bst_nptr = c_bst_tree_next_item( bst_nptr, ascending );				
		}			
	}
			
	return (bst_nptr);
}


bst_tree_node *c_bst_tree_next_item( bst_tree_node *bst_current_nptr, int ascending )
{
	int finished;
	bst_tree_node *bst_nptr, *parent_ptr;
	
	if (bst_current_nptr == NULL)
	{
		printf( "Error: null link passed to bst_next_item()" );
		exit(1);
	}
	else
	{ 
		if (ascending)
		{
			if (bst_current_nptr->lchild != NULL)
				bst_nptr = bst_current_nptr->lchild;
			else
			if (bst_current_nptr->rchild != NULL)
				bst_nptr = bst_current_nptr->rchild;
			else
			{
				bst_nptr = bst_current_nptr;
				
				parent_ptr = bst_nptr->parent_ptr;
				
				finished = false;
				
				while (! finished)
				{
					if (parent_ptr == NULL)
					{
						finished = true;
						bst_nptr = NULL;
					}
					else
					if (parent_ptr->lchild == bst_nptr &&
						parent_ptr->rchild != NULL)
					{
						bst_nptr = parent_ptr->rchild;
						finished = true;
					}
					else
					{
						bst_nptr = parent_ptr;
						parent_ptr = parent_ptr->parent_ptr;
					}
				}
			}
		}
		else
		{
			bst_nptr = bst_current_nptr;
		
			parent_ptr = bst_nptr->parent_ptr;
				
			if (parent_ptr == NULL)
				bst_nptr = NULL;
			else
			{
				if (parent_ptr->rchild == bst_nptr &&
					parent_ptr->lchild != NULL)
				{
					bst_nptr = parent_ptr->lchild;
					
					while (! (bst_nptr->lchild == NULL && bst_nptr->rchild == NULL))
					{
						if (bst_nptr->rchild != NULL)
							bst_nptr = bst_nptr->rchild;
						else   
							bst_nptr = bst_nptr->lchild;   
					}
				}
				else
					bst_nptr = bst_nptr->parent_ptr;
			}
		}
			
		if (bst_nptr != NULL)
		{
			if (! bst_nptr->is_set)
				bst_nptr = c_bst_tree_next_item( bst_nptr, ascending );
		}
	}
		
/*

if full scan of tree

		finished = false;
	
		bst_nptr = bst_current_nptr->next;
				
		while (! finished)
		{
			if (bst_nptr == NULL)
				finished = true;
			else
			if (bst_nptr->is_set)
				finished = true;
			else
				bst_nptr = bst_nptr->next;
		}
	}
*/	
	return (bst_nptr);
}


int bget_bit1( char const *key_bits, int pos )
{
	int num;
	int pos1, bit_pos;

	pos1 = (int) (pos / 8);
	
	bit_pos = pos - (pos1 * 8);
	 		
	if ((key_bits[pos1] & (((int) 0x80) >> bit_pos)) != 0)  
		num = 1;
	else
		num = 0;
	
/*	
	if (bit_pos == 0) {if ((key_bits[pos1] & 0x80) != 0)				num = 1;		}	else
	if (bit_pos == 1) {if ((key_bits[pos1] & 0x40) != 0)				num = 1;		}	else
	if (bit_pos == 2) {if ((key_bits[pos1] & 0x20) != 0)				num = 1;		}	else
	if (bit_pos == 3) {if ((key_bits[pos1] & 0x10) != 0)				num = 1;		}	else
	if (bit_pos == 4) {if ((key_bits[pos1] & 0x08) != 0)				num = 1;		}	else
	if (bit_pos == 5) {if ((key_bits[pos1] & 0x04) != 0)				num = 1;		}	else
	if (bit_pos == 6) {if ((key_bits[pos1] & 0x02) != 0)				num = 1;		}	else
	if (bit_pos == 7) {if ((key_bits[pos1] & 0x01) != 0)				num = 1;		}
*/
	
	return (num);
}


int bget_bit( char const *key_bits, int pos1, int pos2 )
{
	if ((key_bits[pos1] & pos2) != 0)
		return( 1 );
	else
		return( 0 );
}

